"""
Our bound vs gu bound Plot
"""

import matplotlib.pyplot as plt
import numpy as np

data = {
    2: [("Ising Model", 3.7272, 16.9706, 24.0000),("Heisenberg Model", 2.5893, 18.0000, 18.0000),("Mixed-Field Model", 4.0398, 28.3246, 38.4000)],
    4: [("Ising Model", 2.9703, 33.9411, 48.0000), ("Heisenberg Model", 1.9727, 36.0000, 36.0000),("Mixed-Field Model", 3.7227, 56.6493, 76.8000)],
    8: [("Ising Model", 3.0685, 67.8823, 96.0000),("Heisenberg Model", 2.1658, 72.0000, 72.0000),("Mixed-Field Model", 3.6767, 113.2985, 153.6000)]
}

model_order = ["Heisenberg Model", "Ising Model", "Mixed-Field Model"]
n_values = sorted(data.keys()) 
bar_width = 0.20      
group_spacing_factor = 1.5 

model_data = {model: {} for model in model_order}
for n, vals in data.items():
    for model_name, lmax, our_bound, gu_bound in vals:
        if model_name in model_data:
            model_data[model_name][n] = (lmax / our_bound * 100, lmax / gu_bound * 100)

colors = {
    2: ('#4c72b0', '#82a3d1'), 
    4: ('#55a868', '#82c092'), 
    8: ('#8172b2', '#ad9ed1') 
}
hatches = ('///', 'ooo') # Our, Gu


fig, ax = plt.subplots(figsize=(10, 5))
ax.set_facecolor('gainsboro')
ax.grid(True, which='major', linestyle='-', linewidth=1.5, alpha=0.8, zorder=0)

x_base = np.arange(len(model_order)) * group_spacing_factor
num_bar_pairs = len(n_values)
total_group_width = num_bar_pairs * 2 * bar_width

for i, model_name in enumerate(model_order):
    group_start_pos = x_base[i] - total_group_width / 2 + bar_width / 2

    for j, n in enumerate(n_values):
        our_ratio, gu_ratio = model_data[model_name][n]
        our_color, gu_color = colors[n]
        bar_pair_offset = j * 2 * bar_width

        ax.bar(group_start_pos + bar_pair_offset, our_ratio, bar_width,
               color=our_color, hatch=hatches[0], edgecolor='black', zorder=3,
               label=f"$L_{{max}}$ / our bound (n={n})" if i == 0 else "")

        ax.bar(group_start_pos + bar_pair_offset + bar_width, gu_ratio, bar_width,
               color=gu_color, hatch=hatches[1], edgecolor='black', zorder=3,
               label=f"$L_{{max}}$ / Gu bound (n={n})" if i == 0 else "")


ax.set_ylabel(r'$L_{\mathrm{max}} \,/\, L_{\mathrm{upper}}$ (%)', fontsize=20)
ax.tick_params(axis='y', labelsize=16)
ax.set_ylim(0, 25)

ax.set_xticks(x_base)
ax.set_xticklabels(model_order, fontsize=16)
ax.tick_params(axis='x', length=0)
ax.set_xlabel('')

handles, labels = ax.get_legend_handles_labels()
reordered_handles = [handles[0], handles[2], handles[4], handles[1], handles[3], handles[5]]
reordered_labels = [labels[0], labels[2], labels[4], labels[1], labels[3], labels[5]]
fig.legend(reordered_handles, reordered_labels,
           ncol=3, loc='upper center', bbox_to_anchor=(0.5, 1.03),
           frameon=False, fontsize='large')

plt.tight_layout(rect=[0, 0, 1, 0.95])
plt.show()


data = {
    2: [("Ising Model", 3.7272, 16.9706, 24.0000),
        ("Heisenberg Model", 2.5893, 18.0000, 18.0000),
        ("Mixed-Field Model", 4.0398, 28.3246, 38.4000)],
    4: [("Ising Model", 2.9703, 33.9411, 48.0000),
        ("Heisenberg Model", 1.9727, 36.0000, 36.0000),
        ("Mixed-Field Model", 3.7227, 56.6493, 76.8000)],
    8: [("Ising Model", 3.0685, 67.8823, 96.0000),
        ("Heisenberg Model", 2.1658, 72.0000, 72.0000),
        ("Mixed-Field Model", 3.6767, 113.2985, 153.6000)]
}


def compare_ratios(data):
    for n_qubits, entries in data.items():
        print(f"\n=== {n_qubits} qubits ===")
        for obs, Lmax, B1, B2 in entries:
            ratio1 = Lmax / B1 if B1 != 0 else float('inf')
            ratio2 = Lmax / B2 if B2 != 0 else float('inf')

            if ratio1 == ratio2:
                status = "same"
            elif ratio1 < ratio2:
                diff = (1 - ratio1 / ratio2) * 100
                status = f"{diff:.1f}% tighter"
            else:
                diff = (ratio1 / ratio2 - 1) * 100
                status = f"{diff:.1f}% higher"

            print(f"{obs:20}: Lmax/Bound1={ratio1:.4f}, Lmax/Bound2={ratio2:.4f} → {status}")


compare_ratios(data)